#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
using BROKEN = gov.va.med.vbecs.Common.VbecsBrokenRules.FrmMaintainSpecimen;

#endregion
namespace gov.va.med.vbecs.BOL.UnitTests
{

	[TestFixture]
	public class BOL_Specimen : BaseTest 
	{
		private BOL.Specimen _tstSpecimen;
		private DataTable _dtPatientSpecimen;
		private DataTable _dtComment;
		

		[SetUp]
		protected void SetUp() 
		{
			this.BaseSetUp();

			if (this.RefreshData)
			{
			    Guid patientguid = (System.Guid)DataCreator.GetRandomPatientRecord(true).Rows[0][0];
			    this.CreateTestDataSameSpecimen(patientguid, false, true, true, true);
                //AND MaintainableSpecimenIndicator = 1
                this._dtPatientSpecimen = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM PatientSpecimen JOIN OrderedTest ON OrderedTest.PatientSpecimenGuid = PatientSpecimen.PatientSpecimenGuid WHERE PatientSpecimen.PatientGuid = '" + patientguid.ToString() + "' AND SpecimenUid IS NOT NULL AND PatientSpecimen.DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'", true);

                patientguid = (System.Guid)DataCreator.GetRandomPatientRecord(true).Rows[0][0];
			    this.CreateTestDataSameSpecimen(patientguid, true, false, true, true);
                patientguid = (System.Guid)DataCreator.GetRandomPatientRecord(true).Rows[0][0];
			    this.CreateTestDataSameSpecimen(patientguid, false, false, false, true);
                patientguid = (System.Guid)DataCreator.GetRandomPatientRecord(true).Rows[0][0];
			    this.CreateTestDataSameSpecimen(patientguid, true, true, true, true);

				this._dtComment = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM CannedComment WHERE RecordStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'", true);

				this.RefreshData = false;
			}

			this._tstSpecimen = new BOL.Specimen((System.Guid)this._dtPatientSpecimen.Rows[0][TABLES.PatientSpecimen.PatientSpecimenGuid]);
			this._tstSpecimen.SetInitialBrokenRules(BROKEN.FormClassName, BROKEN.RuleSets.ExistingSpecimen);
		}


		[Test]
		public void Ctor_Pass()
		{
			BOL.Specimen tmpS = new BOL.Specimen();

			Assert.IsNotNull(tmpS);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Ctor_Fail()
		{
		}

		[Test]
		public void Ctor_DataRow_Pass()
		{
			BOL.Specimen tmpS = new BOL.Specimen(this._dtPatientSpecimen.Rows[0]);

			Assert.AreEqual(this._dtPatientSpecimen.Rows[0][TABLES.PatientSpecimen.PatientSpecimenGuid].ToString(), tmpS.SpecimenGuid.ToString());
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Ctor_DataRow_Fail()
		{
		}

		[Test]
		public void Ctor_String_Pass()
		{
			BOL.Specimen tmpS = new BOL.Specimen(this._tstSpecimen.SpecimenUid);

			Assert.AreEqual(this._tstSpecimen.IsNew, tmpS.IsNew);
			Assert.AreEqual(this._tstSpecimen.SpecimenUid, tmpS.SpecimenUid);
		}

		[Test]
		public void Ctor_String_Fail()
		{
			string tstValue = "Wiggum";
			BOL.Specimen tmpS = new BOL.Specimen(tstValue);

			Assert.AreEqual(tstValue, tmpS.SpecimenUid);
			Assert.AreEqual(true, tmpS.IsNew);
		}

		[Test]
		public void Ctor_Guid_Pass()
		{
			BOL.Specimen tmpS = new BOL.Specimen(this._tstSpecimen.SpecimenGuid);
			
			Assert.AreEqual(this._tstSpecimen.SpecimenGuid, tmpS.SpecimenGuid, "SpecimenGuid");
			Assert.AreEqual(tmpS.IsNew, false, "IsNew");
		}

		[Test]
		public void Ctor_Guid_Fail()
		{
			System.Guid unitGuid = System.Guid.NewGuid();

			BOL.Specimen tmpS = new BOL.Specimen(unitGuid);

			Assert.AreEqual(true, tmpS.IsNew, "IsNew");
			Assert.AreEqual(unitGuid, tmpS.SpecimenGuid, "SpecimenGuid");
		}

		[Test]
		public void SpecimenGuid_Pass()
		{
			this.RefreshData = true;

			System.Guid tstValue = System.Guid.NewGuid();
			
			this._tstSpecimen.SpecimenGuid = tstValue;
			
			Assert.AreEqual(tstValue, this._tstSpecimen.SpecimenGuid, "SpecimenGuid");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SpecimenGuid_Fail()
		{
		}

		
		[Test]
		public void IsAssociatedWithPendingOrder_Pass()
		{
			this.RefreshData = true;

			bool tstValue = (!this._tstSpecimen.IsAssociatedWithPendingOrder);
			
			this._tstSpecimen.IsAssociatedWithPendingOrder = tstValue;
			
			Assert.AreEqual(tstValue, this._tstSpecimen.IsAssociatedWithPendingOrder, "IsAssociatedWithPendingOrder");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IsAssociatedWithPendingOrder_Fail()
		{
		}
		
		[Test]
		public void Patient_Pass()
		{
			this.RefreshData = true;

			BOL.Patient tstValue = new BOL.Patient();

			this._tstSpecimen.Patient = tstValue;
			Assert.IsTrue(tstValue == this._tstSpecimen.Patient);
		}

		[Test]
		public void Patient_With_ConversionTRs_Pass()
		{

            Guid patientGuid = (Guid)DataCreator.GetRandomPatientRecord(true).Rows[0][0];
			BOL.Patient patient = new BOL.Patient(patientGuid);
			Guid TRGuidFromConv = Guid.NewGuid();
			
			string insertTransfusionReqFromConv = "INSERT INTO PATIENTTRANSFUSIONREQUIREMENT (";
			insertTransfusionReqFromConv += TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid + ",";
			insertTransfusionReqFromConv += TABLES.PatientTransfusionRequirement.PatientGuid + ",";
			insertTransfusionReqFromConv += TABLES.PatientTransfusionRequirement.TransfusionRequirementText + ",";
			insertTransfusionReqFromConv += TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + ",";
			insertTransfusionReqFromConv += TABLES.PatientTransfusionRequirement.CreateDate + ",";
			insertTransfusionReqFromConv += TABLES.PatientTransfusionRequirement.CreateUser + ",";
			insertTransfusionReqFromConv += TABLES.PatientTransfusionRequirement.CreateDivisionCode + ",";
			insertTransfusionReqFromConv += TABLES.PatientTransfusionRequirement.VistaConversionIndicator + ",";
			insertTransfusionReqFromConv += TABLES.PatientTransfusionRequirement.AntibodyTypeId + ",";
			insertTransfusionReqFromConv += TABLES.PatientTransfusionRequirement.RecordStatusCode + ",";
			insertTransfusionReqFromConv += TABLES.PatientTransfusionRequirement.LastUpdateDate + ",";
			insertTransfusionReqFromConv += TABLES.PatientTransfusionRequirement.LastUpdateUser + ",";
			insertTransfusionReqFromConv += TABLES.PatientTransfusionRequirement.LastUpdateFunctionId + ")";
			insertTransfusionReqFromConv += "VALUES (";
			insertTransfusionReqFromConv += "'" + TRGuidFromConv.ToString() + "',";
			insertTransfusionReqFromConv += "'" + patientGuid.ToString() + "',";
			insertTransfusionReqFromConv += "'Anti-C',";
			insertTransfusionReqFromConv += "'AS',";
			insertTransfusionReqFromConv += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertTransfusionReqFromConv += "'" + Environment.UserName + "',";
			insertTransfusionReqFromConv += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertTransfusionReqFromConv += "'1',";
			insertTransfusionReqFromConv += "'5',";
			insertTransfusionReqFromConv += "'C',";  //TR from conversion
			insertTransfusionReqFromConv += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertTransfusionReqFromConv += "'" + Environment.UserName + "',";
			insertTransfusionReqFromConv += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			UnitTestUtil.RunSQL(insertTransfusionReqFromConv);

			this._tstSpecimen.Patient = patient;
			Assert.IsTrue(this._tstSpecimen.HasAntibodyIdentification == true);
		
			string deleteTRs = "DELETE FROM PATIENTTRANSFUSIONREQUIREMENT WHERE PatientTransfusionRequirementGuid = '" + TRGuidFromConv.ToString() + "'";
			UnitTestUtil.RunSQL(deleteTRs);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Patient_Fail()
		{
		}
		
		[Test]
		public void SpecimenAccessionNumber_Pass()
		{
			this.RefreshData = true;

			string tstValue = "BB 0608 15";

			this._tstSpecimen.SpecimenAccessionNumber = tstValue;

			Assert.AreEqual(tstValue, this._tstSpecimen.SpecimenAccessionNumber);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SpecimenAccessionNumber_Fail()
		{
		}
		
		[Test]
		public void SpecimenUid_Pass()
		{
			this.RefreshData = true;

			string tstValue = "123456790123";

			this._tstSpecimen.SpecimenUid = tstValue;

			Assert.AreEqual(tstValue, this._tstSpecimen.SpecimenUid, "SpecimenUid");
			Assert.AreEqual(false, this._tstSpecimen.IsRuleBroken(BROKEN.SpecimenUid), "IsRuleBroken");
		}

		[Test]
		public void SpecimenUid_Fail()
		{
			this.RefreshData = true;

			this._tstSpecimen.SpecimenUid = "".PadRight(9,'X');
			Assert.AreEqual(true, this._tstSpecimen.IsRuleBroken(BROKEN.SpecimenUid), "IsRuleBroken-1");

			this._tstSpecimen.SpecimenUid = string.Empty;
			Assert.AreEqual(true, this._tstSpecimen.IsRuleBroken(BROKEN.SpecimenUid), "IsRuleBroken-2");

			this._tstSpecimen.SpecimenUid = "".PadRight(16,'X');
			Assert.AreEqual(true, this._tstSpecimen.IsRuleBroken(BROKEN.SpecimenUid), "IsRuleBroken-3");
		}

		[Test]
		public void VBECSSpecimenNumber_Pass()
		{
			this.RefreshData = true;

			string tstValue = System.DateTime.Now.ToString();

			this._tstSpecimen.VBECSSpecimenNumber = tstValue;

			Assert.AreEqual(tstValue, this._tstSpecimen.VBECSSpecimenNumber);
		}
		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void VBECSSpecimenNumber_Fail()
		{
		}

		[Test]
		public void SpecimenCollectionDate_Pass()
		{
			this.RefreshData = true;

			System.DateTime tstValue = System.DateTime.Now.AddMinutes(-1);
			this._tstSpecimen.SpecimenCollectionDate = tstValue;
			this._tstSpecimen.SpecimenReceivedDate = System.DateTime.Now;
			this._tstSpecimen.SpecimenExpirationDate = System.DateTime.Now.AddDays(1);

			Assert.AreEqual(tstValue.ToString(), this._tstSpecimen.SpecimenCollectionDate.ToString(), "SpecimenCollectionDate");
			Assert.AreEqual(false, this._tstSpecimen.IsRuleBroken(BROKEN.SpecimenCollectionDate), "IsRuleBroken");
		}

		[Test]
		public void SpecimenCollectionDate_Fail()
		{
			this.RefreshData = true;

			this._tstSpecimen.SpecimenCollectionDate = DateTime.Now.AddDays(1);
			Assert.AreEqual(true, this._tstSpecimen.IsRuleBroken(BROKEN.SpecimenCollectionDate), "IsRuleBroken-1");

			this._tstSpecimen.SpecimenCollectionDate = DateTime.MinValue;
			Assert.AreEqual(true, this._tstSpecimen.IsRuleBroken(BROKEN.SpecimenCollectionDate), "IsRuleBroken-2");
		}

		[Test]
		public void SpecimenReceivedDate_Pass()
		{
			this.RefreshData = true;

			System.DateTime tstValue = System.DateTime.Now.AddMinutes(-1);
			this._tstSpecimen.SpecimenCollectionDate = tstValue.AddHours(-1);
			this._tstSpecimen.SpecimenReceivedDate = tstValue;
			this._tstSpecimen.SpecimenExpirationDate = System.DateTime.Now.AddDays(1);

			Assert.AreEqual(tstValue, this._tstSpecimen.SpecimenReceivedDate, "SpecimenReceivedDate");
			Assert.AreEqual(false, this._tstSpecimen.IsRuleBroken(BROKEN.SpecimenReceivedDate), "IsRuleBroken");
		}

		[Test]
		public void SpecimenReceivedDate_Fail()
		{
			this.RefreshData = true;

			this._tstSpecimen.SpecimenReceivedDate = DateTime.Now.AddDays(1);
			Assert.AreEqual(true, this._tstSpecimen.IsRuleBroken(BROKEN.SpecimenReceivedDate), "IsRuleBroken-1");

			this._tstSpecimen.SpecimenReceivedDate = DateTime.MinValue;
			Assert.AreEqual(true, this._tstSpecimen.IsRuleBroken(BROKEN.SpecimenReceivedDate), "IsRuleBroken-2");

			this._tstSpecimen.SpecimenReceivedDate = this._tstSpecimen.SpecimenCollectionDate.AddDays(-1);
			Assert.AreEqual(true, this._tstSpecimen.IsRuleBroken(BROKEN.SpecimenReceivedDate), "IsRuleBroken-3");

		}

		[Test]
		public void SpecimenSource_Pass()
		{
			this.RefreshData = true;

			string tstValue = System.DateTime.Now.ToString();

			this._tstSpecimen.SpecimenSource = tstValue;

			Assert.AreEqual(tstValue, this._tstSpecimen.SpecimenSource);
		}
		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SpecimenSource_Fail()
		{
		}

		[Test]
		public void Phlebotomist_Pass()
		{
			this.RefreshData = true;

			string tstValue = "".PadRight(35,'X');

			this._tstSpecimen.Phlebotomist = tstValue;
			Assert.AreEqual(tstValue, this._tstSpecimen.Phlebotomist, "Phlebotomist");
			Assert.AreEqual(false, this._tstSpecimen.IsRuleBroken(BROKEN.Phlebotomist), "IsRuleBroken");
		}

		[Test]
		public void Phlebotomist_Fail()
		{
			this.RefreshData = true;

			this._tstSpecimen.Phlebotomist = "".PadRight(36,'X');
			Assert.AreEqual(true, this._tstSpecimen.IsRuleBroken(BROKEN.Phlebotomist), "IsRuleBroken-1");

			this._tstSpecimen.Phlebotomist = "XX";
			Assert.AreEqual(true, this._tstSpecimen.IsRuleBroken(BROKEN.Phlebotomist), "IsRuleBroken-2");
		}

		[Test]
		public void MaintainableSpecimenIndicator_Pass()
		{
			this.RefreshData = true;

			bool tstValue = (!this._tstSpecimen.MaintainableSpecimenIndicator);

			this._tstSpecimen.MaintainableSpecimenIndicator = tstValue;

			Assert.AreEqual(tstValue, this._tstSpecimen.MaintainableSpecimenIndicator, "MaintainableSpecimenIndicator");
		}
		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void MaintainableSpecimenIndicator_Fail()
		{
		}

		[Test]
		public void SpecimenExpirationDate_Pass()
		{
			this.RefreshData = true;

			System.DateTime tstValue = System.DateTime.Now;

			this._tstSpecimen.SpecimenExpirationDate = tstValue;

			Assert.AreEqual(tstValue, this._tstSpecimen.SpecimenExpirationDate, "SpecimenExpirationDate");
		}

		[Test]
		public void SpecimenExpirationDateDefault_Pass()
		{
			BOL.Specimen specimen = new BOL.Specimen();

			System.DateTime tstValue = System.DateTime.Now;

			specimen.SpecimenCollectionDate = tstValue;

			Assert.AreEqual(tstValue.Date.AddDays(3).AddHours(23).AddMinutes(59), specimen.SpecimenExpirationDate, "SpecimenExpirationDate");
		}

		[Test]
		public void SpecimenExpirationDate_Fail()
		{
			this.RefreshData = true;

			this._tstSpecimen.SpecimenExpirationDate = this._tstSpecimen.SpecimenCollectionDate.AddDays(-1);
			Assert.AreEqual(true, this._tstSpecimen.IsRuleBroken(BROKEN.SpecimenExpirationDate), "SpecimenExpirationDate");
		}

		[Test]
		public void SpecimenStatusCode_Pass()
		{
			this.RefreshData = true;

			string tstValue = System.DateTime.Now.ToString();

			this._tstSpecimen.SpecimenStatusCode = tstValue;

			Assert.AreEqual(tstValue, this._tstSpecimen.SpecimenStatusCode, "SpecimenStatusCode");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SpecimenStatusCode_Fail()
		{
		}

		[Test]
		public void UnacceptableSpecimenIndicator_Pass()
		{
			this.RefreshData = true;

			this._tstSpecimen.IsAssociatedWithPendingOrder = false;
			this._tstSpecimen.UnacceptableSpecimenIndicator = true;

			Assert.AreEqual(true, this._tstSpecimen.UnacceptableSpecimenIndicator, "UnacceptableSpecimenIndicator");
			Assert.AreEqual(false, this._tstSpecimen.IsRuleBroken(BROKEN.UnacceptableSpecimenIndicator), "IsRuleBroken");
		}

		[Test]
		public void UnacceptableSpecimenIndicator_Fail()
		{
			this.RefreshData = true;

			this._tstSpecimen.IsAssociatedWithPendingOrder = true;
			this._tstSpecimen.UnacceptableSpecimenIndicator = true;
			Assert.AreEqual(true, this._tstSpecimen.IsRuleBroken(BROKEN.UnacceptableSpecimenIndicator), "UnacceptableSpecimenIndicator");
		}

		[Test]
		public void PostTransfusionIndicator_Pass()
		{
			this.RefreshData = true;

			bool tstValue = (!this._tstSpecimen.PostTransfusionIndicator);

			this._tstSpecimen.PostTransfusionIndicator = tstValue;

			Assert.AreEqual(tstValue, this._tstSpecimen.PostTransfusionIndicator, "PostTransfusionIndicator");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void PostTransfusionIndicator_Fail()
		{
		}

		[Test]
		public void DivisionCode_Pass()
		{
			this.RefreshData = true;

			string tstValue = System.DateTime.Now.ToString();

			this._tstSpecimen.DivisionCode = tstValue;

			Assert.AreEqual(tstValue, this._tstSpecimen.DivisionCode, "DivisionCode");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void DivisionCode_Fail()
		{
		}

		[Test]
		public void ProcessingTechId_Pass()
		{
			this.RefreshData = true;

			string tstValue = System.DateTime.Now.ToString();

			this._tstSpecimen.ProcessingTechId = tstValue;

			Assert.AreEqual(tstValue, this._tstSpecimen.ProcessingTechId, "ProcessingTechId");
			Assert.AreEqual(false,  this._tstSpecimen.IsRuleBroken(BROKEN.ProcessingTechId), "IsRuleBroken");
		}

		[Test]
		public void ProcessingTechId_Fail()
		{
			this.RefreshData = true;

			this._tstSpecimen.ProcessingTechId = string.Empty;
			Assert.AreEqual(true, this._tstSpecimen.IsRuleBroken(BROKEN.ProcessingTechId), "IsRuleBroken");
		}

		[Test]
		public void UnacceptableSpecimenComment_Pass()
		{
			this.RefreshData = true;

			BOL.CannedComment tmpC = new BOL.CannedComment((System.Guid)this._dtComment.Rows[0][TABLES.CannedComment.CannedCommentGuid]);

			this._tstSpecimen.UnacceptableSpecimenIndicator = false;
			this._tstSpecimen.UnacceptableSpecimenComment = tmpC;
			Assert.AreEqual(false, this._tstSpecimen.IsRuleBroken(BROKEN.CannedCommentGuid), "IsRuleBroken");
		}

		[Test]
		public void UnacceptableSpecimenComment_Fail()
		{
			this.RefreshData = true;

			BOL.CannedComment tmpC = new BOL.CannedComment(System.Guid.Empty);
			this._tstSpecimen.UnacceptableSpecimenIndicator = true;

			this._tstSpecimen.UnacceptableSpecimenComment = tmpC;
			Assert.AreEqual(true, this._tstSpecimen.IsRuleBroken(BROKEN.CannedCommentGuid), "IsRuleBroken");
		}

		[Test]
		public void UnacceptableSpecimenDetails_Pass()
		{
			this.RefreshData = true;

			string tstValue = System.DateTime.Now.ToString();

			this._tstSpecimen.UnacceptableSpecimenDetails = tstValue;
			Assert.AreEqual(tstValue, this._tstSpecimen.UnacceptableSpecimenDetails, "UnacceptableSpecimenDetails");
			Assert.AreEqual(false, this._tstSpecimen.IsRuleBroken(BROKEN.Details), "IsRuleBroken");
		}

		[Test]
		public void UnacceptableSpecimenDetails_Fail()
		{
			this.RefreshData = true;

			BOL.CannedComment tmpC = new BOL.CannedComment((System.Guid)this._dtComment.Rows[0][TABLES.CannedComment.CannedCommentGuid]);

			tmpC.Comments = "OTHER";
			this._tstSpecimen.UnacceptableSpecimenComment = tmpC;
			this._tstSpecimen.UnacceptableSpecimenIndicator = true;
			this._tstSpecimen.UnacceptableSpecimenDetails = string.Empty;
			
			Assert.AreEqual(true, this._tstSpecimen.IsRuleBroken(BROKEN.Details), "IsRuleBroken");
		}

		[Test]
		public void IsExpired_Pass()
		{
			bool tstValue = this._tstSpecimen.IsExpired;

			Assert.AreEqual(tstValue, this._tstSpecimen.IsExpired, "IsExpired");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void IsExpired_Fail()
		{
		}

		[Test]
		public void IsSpecimenUidComplete_Pass()
		{
			bool tstValue = this._tstSpecimen.IsSpecimenUidComplete;

			Assert.AreEqual(tstValue, this._tstSpecimen.IsSpecimenUidComplete, "IsSpecimenUidComplete");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void IsSpecimenUidComplete_Fail()
		{
		}

		[Test]
		public void IsExtended_Pass()
		{
			bool tstValue = this._tstSpecimen.IsExtended;

			Assert.AreEqual(tstValue, this._tstSpecimen.IsExtended, "IsExtended");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void IsExtended_Fail()
		{
		}

		[Test]
		public void MaximumSpecimenExpirationDays_Pass()
		{
			this.RefreshData = true;

			short tstValue = 4;

			this._tstSpecimen.MaximumSpecimenExpirationDays = tstValue;

			Assert.AreEqual(tstValue, this._tstSpecimen.MaximumSpecimenExpirationDays, "MaximumSpecimenExpirationDays");
			Assert.AreEqual(false, this._tstSpecimen.IsRuleBroken(BROKEN.MaximumSpecimenExpirationDays), "IsRuleBroken");
		}

		[Test]
		public void MaximumSpecimenExpirationDays_Fail()
		{
			this.RefreshData = true;

			this._tstSpecimen.MaximumSpecimenExpirationDays = 1;
			Assert.AreEqual(true, this._tstSpecimen.IsRuleBroken(BROKEN.MaximumSpecimenExpirationDays), "IsRuleBroken-1");

			this._tstSpecimen.MaximumSpecimenExpirationDays = 999;
			Assert.AreEqual(true, this._tstSpecimen.IsRuleBroken(BROKEN.MaximumSpecimenExpirationDays), "IsRuleBroken-2");
		}

		[Test]
		public void TestResults_Pass()
		{
			DataTable tstValue = this._tstSpecimen.TestResults;

			Assert.AreEqual(this._tstSpecimen.TestResults.Rows.Count, tstValue.Rows.Count, "Rows.Count");
			Assert.AreEqual(this._tstSpecimen.TestResults.Columns.Count, tstValue.Columns.Count, "Columns.Count");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void TestResults_Fail()
		{
		}

		[Test]
		public void HasAntibodyIdentification_Pass()
		{
			bool tstValue = this._tstSpecimen.HasAntibodyIdentification;

			Assert.AreEqual( tstValue, this._tstSpecimen.HasAntibodyIdentification, "HasAntibodyIdentification");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void HasAntibodyIdentification_Fail()
		{
		}

		[Test]
		public void HasTransfusionHistory_Pass()
		{
			bool tstValue = this._tstSpecimen.HasTransfusionHistory;

			Assert.AreEqual(tstValue, this._tstSpecimen.HasTransfusionHistory, "HasTransfusionHistory");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void HasTransfusionHistory_Fail()
		{
		}

		[Test]
		public void HasAssignedUnits_Pass()
		{
			bool tstValue = this._tstSpecimen.HasAssignedUnits;

			Assert.AreEqual(tstValue, this._tstSpecimen.HasAssignedUnits, "HasAssignedUnits");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void HasAssignedUnits_Fail()
		{
		}

		[Test]
		public void PreviouslySavedAsUnacceptable_Pass()
		{
			bool tstValue = this._tstSpecimen.PreviouslySavedAsUnacceptable;

			Assert.AreEqual(tstValue, this._tstSpecimen.PreviouslySavedAsUnacceptable, "PreviouslySavedAsUnacceptable");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void PreviouslySavedAsUnacceptable_Fail()
		{
		}

		[Test]
		public void PreviousExpirationDate_Pass()
		{
			this.RefreshData = true;

			System.DateTime tstValue = System.DateTime.Now;
				
			this._tstSpecimen.PreviousExpirationDate = tstValue;

			Assert.AreEqual(tstValue, this._tstSpecimen.PreviousExpirationDate, "PreviousExpirationDate");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void PreviousExpirationDate_Fail()
		{
		}

		[Test]
		public void ABORhTested_CompleteResults_Pass()
		{
            Guid patientGuid = (Guid)DataCreator.GetRandomPatientRecord(true).Rows[0][0];
            // Clean up patient data
            ClearPatientData(patientGuid);

            bool testComplete = TestABORhResultsSpecimen(patientGuid, true); // ABO and RH test both complete
            Assert.IsTrue(testComplete);
		}

        [Test]
        public void ABORhTested_ExtraResults_Pass()
        {
            Guid patientGuid = (Guid)DataCreator.GetRandomPatientRecord(true).Rows[0][0];
            // Clean up patient data
            ClearPatientData(patientGuid);
            Guid psGuid = System.Guid.NewGuid();

            Guid otGuid = DataCreator.GetRecordGuid("SELECT TOP 1 OrderedTestGuid FROM OrderedTest");   //im gettin lazy
            Guid st1Guid = System.Guid.NewGuid();
            Guid st2Guid = System.Guid.NewGuid();
            Guid st3Guid = System.Guid.NewGuid();
            Guid st4Guid = System.Guid.NewGuid();

            ArrayList dtArray = new ArrayList();
            dtArray.Add(SprocHelper.GetInsertPatientSpecimenSprocTable(psGuid, patientGuid, DataCreator.GetNewSpecimenUid()));
            dtArray.Add(SprocHelper.GetInsertSpecimenTestSprocTable(st1Guid, otGuid, psGuid, (int) Common.TestType.ABOInterp));
            dtArray.Add(SprocHelper.GetInsertSpecimenTestSprocTable(st2Guid, otGuid, psGuid, (int)Common.TestType.RhInterp));
            dtArray.Add(SprocHelper.GetInsertSpecimenTestSprocTable(st3Guid, otGuid, psGuid, (int)Common.TestType.ABOInterpTAS));
            dtArray.Add(SprocHelper.GetInsertSpecimenTestSprocTable(st4Guid, otGuid, psGuid, (int)Common.TestType.RhInterpTAS));

            ArrayList sprocArray = new ArrayList();
            sprocArray.Add(SPROCS.InsertPatientSpecimen.StoredProcName);
            sprocArray.Add(SPROCS.InsertSpecimenTest.StoredProcName);
            sprocArray.Add(SPROCS.InsertSpecimenTest.StoredProcName);
            sprocArray.Add(SPROCS.InsertSpecimenTest.StoredProcName);
            sprocArray.Add(SPROCS.InsertSpecimenTest.StoredProcName);

            new Common.StoredProcedure().TransactionalGetValue(sprocArray, dtArray);
            BOL.Specimen tstValue = new Specimen(psGuid);
            Assert.IsFalse(tstValue.ABORhTested);

            UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid = '" + st1Guid.ToString() + "'");
            UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid = '" + st2Guid.ToString() + "'");
            UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid = '" + st3Guid.ToString() + "'");
            UnitTestUtil.RunSQL("DELETE FROM SpecimenTest WHERE SpecimenTestGuid = '" + st4Guid.ToString() + "'");
            UnitTestUtil.RunSQL("DELETE FROM PatientSpecimen WHERE PatientSpecimenGuid = '" + psGuid.ToString() + "'");
        }

        [Test]
        public void ABORhTested_NoInterps_Pass()
        {
            Guid patientGuid = (Guid)DataCreator.GetRandomPatientRecord(true).Rows[0][0];
            ClearPatientData(patientGuid);

            bool testComplete = TestABORhResultsSpecimen(patientGuid, true); // ABO and RH test both complete
            Assert.IsTrue(testComplete);
        }

        [Test]
        public void ABORhTested_IncompleteResults_Pass()
        {
            Guid patientGuid = (Guid)DataCreator.GetRandomPatientRecord(true).Rows[0][0];
            // Clean up patient data
            ClearPatientData(patientGuid);

            bool testIncomplete = TestABORhResultsSpecimen(patientGuid, false);// ABO complete, but not Rh
            Assert.IsTrue(testIncomplete);
        }

		[Test]
		public void ABORhTested_NoResults_Fail()
		{
            BOL.Specimen tmpS = new BOL.Specimen();
            Assert.IsFalse(tmpS.ABORhTested);
		}

		[Test]
		public void TASTested_Pass()
		{
			bool tstValue = this._tstSpecimen.TASTested;

			Assert.AreEqual(tstValue, this._tstSpecimen.TASTested, "TASTested");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void TASTested_Fail()
		{
		}

		[Test]
		public void UpdateSpecimen_Pass()
		{
			this.RefreshData = true;

			int retValue = this._tstSpecimen.UpdateSpecimen(null, null, Common.UpdateFunction.UnitTests);

			Assert.IsTrue(retValue == 0, "ReturnValue");
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(Common.RowVersionException))]
		public void UpdateSpecimen_Fail()
		{
			this.RefreshData = true;

			this._tstSpecimen.RowVersion = new byte[8];

			this._tstSpecimen.UpdateSpecimen(null,null,Common.UpdateFunction.UnitTests);
		}

		[Test]
		public void UpdateSpecimen_Extended_Pass()
		{
			this.RefreshData = true;

			BOL.ExceptionReport specimenExtendedException = new BOL.ExceptionReport(Common.ExceptionType.Specimen_expiration_date_extended, "Unit Test 7549", Environment.UserName);
			specimenExtendedException.GenerateExceptionSpecimen(null, this._tstSpecimen);
			int retValue = this._tstSpecimen.UpdateSpecimen(null, specimenExtendedException, Common.UpdateFunction.UnitTests);
			//
			Assert.IsTrue(retValue == 0, "ReturnValue");
		}

		[Test]
		public void UpdateSpecimen_Reactivated_Pass()
		{
			this.RefreshData = true;

			BOL.ExceptionReport specimenReactivatedException = new BOL.ExceptionReport(Common.ExceptionType.Unacceptable_Expired_specimen_used, "Unit Test 7550", Environment.UserName);
			specimenReactivatedException.GenerateExceptionSpecimen(null, this._tstSpecimen);
			int retValue = this._tstSpecimen.UpdateSpecimen(specimenReactivatedException, null, Common.UpdateFunction.UnitTests);
			//
			Assert.IsTrue(retValue == 0, "ReturnValue");
		}

		[Test]
		public void InsertSpecimen_Pass()
		{
			BOL.Specimen tmpS = new BOL.Specimen(System.Guid.NewGuid());
			
			tmpS.DivisionCode = UnitTestConfig.TestDivisionCode;
			tmpS.Patient = this._tstSpecimen.Patient;
		    tmpS.SpecimenUid = DataCreator.GetNewSpecimenUid();

			BOL.ExceptionReport specimenExtendedException = new BOL.ExceptionReport(Common.ExceptionType.Specimen_expiration_date_extended, "Unit Test 3554", Environment.UserName);
			specimenExtendedException.GenerateExceptionSpecimen(null, tmpS);
			int retValue = tmpS.InsertSpecimen(specimenExtendedException, Common.UpdateFunction.UnitTests);

			Assert.AreEqual(0, retValue, "ReturnValue");
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(Common.DataAccessLayerException))]
		public void InsertSpecimen_Fail()
		{
			this.RefreshData = true;

			this._tstSpecimen.InsertSpecimen(null,Common.UpdateFunction.UnitTests);
		}

		[Test]
		public void GetSpecimenSaveDataset_Pass()
		{
			DataSet dsTest = this._tstSpecimen.GetSpecimenSaveDataset();

			Assert.AreEqual(1, dsTest.Tables.Count, "Tables.Count");
			Assert.AreEqual(true, dsTest.Tables[0].Rows.Count >= 0, "Tables[0].Rows.Count");
			Assert.AreEqual(19, dsTest.Tables[0].Columns.Count, "Tables[0].Columns.Count");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetSpecimenSaveDataset_Fail()
		{
		}

		[Test]
		public void GetTestResultsForSpecimen_Pass()
		{
			DataTable dtTest = this._tstSpecimen.GetTestResultsForSpecimen();

			Assert.IsTrue(dtTest.Rows.Count >= 0, "Rows.Count");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetTestResultsForSpecimen_Fail()
		{
		}

		[Test]
		public void LoadDataRowFromThis_Pass()
		{
			DataRow dr = this._tstSpecimen.LoadDataRowFromThis(this._dtPatientSpecimen.NewRow());

			Assert.IsNotNull(dr);
			Assert.AreEqual((System.Guid)dr[TABLES.PatientSpecimen.PatientSpecimenGuid] , this._tstSpecimen.SpecimenGuid, "PatientSpecimenGuid");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LoadDataRowFromThis_Fail()
		{
		}


		[Test]
		public void GetSpecimenSearchDetails_Pass()
		{
			DataTable dtTest = BOL.Specimen.GetSpecimenSearchDetails(this._tstSpecimen.SpecimenUid, string.Empty, string.Empty);
			Assert.IsTrue(dtTest.Rows.Count > 0, "Rows.Count");
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(NoRecordsFoundException))]
		public void GetSpecimenSearchDetails_Fail()
		{
			BOL.Specimen.GetSpecimenSearchDetails("Homer", "Jay", "Simpson");
		}


		[Test]
		public void GetSpecimenHistory_Pass()
		{
			DataTable dtTest = BOL.Specimen.GetSpecimenHistory(this._tstSpecimen.SpecimenGuid);
			Assert.IsTrue(dtTest.Rows.Count >= 0, "Rows.Count");
		}

		[Test]
		public void GetSpecimenHistory_Fail()
		{
			DataTable dtTest = BOL.Specimen.GetSpecimenHistory(System.Guid.NewGuid());
			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}
		
		[Test]
		public void GetSpecimenList_Pass()
		{
			string specimenUid = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 SpecimenUid FROM PatientSpecimen WHERE SpecimenUid IS NOT NULL AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "' ORDER BY SpecimenExpirationDate DESC")[TABLES.PatientSpecimen.SpecimenUid].ToString();
			System.DateTime expirationDate = System.DateTime.Now.AddYears(-1);
		    UnitTestUtil.RunSQL("UPDATE PatientSpecimen SET SpecimenExpirationDate = '" + DateTime.Now.AddDays(2).Date.AddHours(23).AddMinutes(59).ToString() + "' WHERE SpecimenUid = '" + specimenUid + "'"); //100% pass!
			DataTable dtTest = BOL.Specimen.GetSpecimenList(specimenUid, string.Empty, string.Empty, expirationDate);
			Assert.IsTrue(dtTest.Rows.Count > 0, "Rows.Count");
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(NoRecordsFoundException))]
		public void GetSpecimenList_Fail()
		{
			BOL.Specimen.GetSpecimenList("Homer", "Jay", "Simpson", System.DateTime.Now.AddYears(-1));
		}

		[Test]
		public void GetSpecimen_Pass()
		{
			DataTable dtTest = BOL.Specimen.GetSpecimen(this._tstSpecimen.SpecimenUid);
			Assert.IsTrue(dtTest.Rows.Count > 0, "Rows.Count");
		}

		[Test]
		public void GetSpecimen_Fail()
		{
			DataTable dtTest = BOL.Specimen.GetSpecimen("VA RULES!");
			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}


		[Test]
		public void CompareDates_Pass()
		{
			this.RefreshData = true;

			string collectionDateError = "";
			string receivedDateError = "";
			string expirationDateError = "";
			//
			DateTime divisionNow = BOL.VBECSDateTime.GetDivisionCurrentDateTime(UnitTestConfig.TestDivisionCode);

			this._tstSpecimen.SpecimenCollectionDate = divisionNow.AddMinutes(1);
			this._tstSpecimen.SpecimenReceivedDate = divisionNow.AddMinutes(1);
			this._tstSpecimen.SpecimenExpirationDate = divisionNow.AddDays(3);
			this._tstSpecimen.CompareDates(ref collectionDateError, ref receivedDateError, ref expirationDateError);
			Assert.IsTrue(collectionDateError.Length>0, "Future Collection Date");
			Assert.IsTrue(receivedDateError.Length>0, "Future Received Date");
			//
			this._tstSpecimen.SpecimenCollectionDate = divisionNow;
			this._tstSpecimen.SpecimenReceivedDate = divisionNow.AddMinutes(-2);
			this._tstSpecimen.SpecimenExpirationDate = divisionNow.AddMinutes(-1);
			this._tstSpecimen.CompareDates(ref collectionDateError, ref receivedDateError, ref expirationDateError);
			Assert.IsTrue(receivedDateError.Length>0, "Received<Collection");
			Assert.IsTrue(expirationDateError.Length>0, "Expiration<Collection");
			//
			this._tstSpecimen.SpecimenCollectionDate = divisionNow.AddMinutes(-2);
			this._tstSpecimen.SpecimenReceivedDate = divisionNow;
			this._tstSpecimen.SpecimenExpirationDate = divisionNow.AddMinutes(-1);
			this._tstSpecimen.CompareDates(ref collectionDateError, ref receivedDateError, ref expirationDateError);
			Assert.IsTrue(expirationDateError.Length>0, "Expiration<Received");
			//
			this._tstSpecimen.SpecimenCollectionDate = divisionNow.AddMinutes(-2);
			this._tstSpecimen.SpecimenReceivedDate = divisionNow.AddMinutes(-1);
			this._tstSpecimen.SpecimenExpirationDate = divisionNow.AddMonths(6);
			this._tstSpecimen.CompareDates(ref collectionDateError, ref receivedDateError, ref expirationDateError);
			Assert.IsTrue(expirationDateError.Length>0, "Expiration>MaxDate");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void CompareDates_Fail()
		{
		}

		[Test]
		public void RepeatABORhTestsMatch_Pass()
		{	
			bool passed = false;

            Guid patientGuid = (Guid)DataCreator.GetRandomPatientRecord(true).Rows[0][0];
			// Clean up patient data
			ClearPatientData(patientGuid);

            passed = CreateTestDataSameSpecimen(patientGuid, true, false, true, false);
			Assert.IsTrue(passed);

            passed = CreateTestDataSameSpecimen(patientGuid, false, true, true, false);
			Assert.IsTrue(passed);

			passed = CreateTestDataSameSpecimen(patientGuid, false, false, false, false);
			Assert.IsTrue(passed);
			//no RC
		}

		[Test]
		public void RepeatABORhTestsMatch_Fail()
		{
			bool passed = false;

            Guid patientGuid = (Guid)DataCreator.GetRandomPatientRecord(true).Rows[0][0];
			// Clean up patient data
			ClearPatientData(patientGuid);

            passed = CreateTestDataSameSpecimen(patientGuid, false, false, false, false);
			Assert.IsTrue(passed, "not justified");

            passed = CreateTestDataSameSpecimen(patientGuid, false, false, true, false);
			Assert.IsTrue(passed, "rh tests");
		}

		[Test]
		public void ABORhTestsMatchHistoricResults_Pass()
		{
			bool passed = false;
            Guid patientGuid = (Guid)DataCreator.GetRandomPatientRecord(true).Rows[0][0];
			// Clean up patient data
			ClearPatientData(patientGuid);

			// match, justified, past ABO justified, past Rh justified
			passed = CreateTestDataDifferentSpecimens(patientGuid, true, false, false, false, false);
			Assert.IsTrue(passed);

            passed = CreateTestDataDifferentSpecimens(patientGuid, false, true, false, false, false);
			Assert.IsTrue(passed);


            passed = CreateTestDataDifferentSpecimens(patientGuid, false, false, true, false, false);
			Assert.IsTrue(passed);

            passed = CreateTestDataDifferentSpecimens(patientGuid, false, false, false, true, false);
			Assert.IsTrue(passed);

			CheckInconclusives(patientGuid, true, false, false, false);
		}

		[Test]
		public void ABORhTestsMatchHistoricResults_Fail()
		{
            Guid patientGuid = (Guid)DataCreator.GetRandomPatientRecord(true).Rows[0][0];
			// Clean up patient data
			ClearPatientData(patientGuid);

            bool passed = CreateTestDataDifferentSpecimens(patientGuid, false, false, false, false, false);
			Assert.IsTrue(passed);
		}

		private void ClearPatientData(Guid patientGuid)
		{
			string sql = "DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + patientGuid + "'";
			UnitTestUtil.RunSQL(sql);

			sql = "DELETE FROM SpecimenTest WHERE SpecimenTestGuid IN " + 
			"(SELECT SpecimenTestGuid FROM SpecimenTest " +
			"JOIN PatientSpecimen PS ON PS.PatientSpecimenGuid = SpecimenTest.PatientSpecimenGuid " +
			"WHERE PS.PatientGuid =  '" + patientGuid + "')";

			UnitTestUtil.RunSQL(sql);
		}

		private bool CreateTestDataSameSpecimen(Guid patientGuid, bool matching, bool currentJustified, bool createRhTests, bool skipDelete)
		{
			bool passed = false;
			// Create two specimen tests to be used to justify abo/rh change
			#region Create Test Data

			// PatientTreatment
			Guid patientTreatmentTestOneGuid = Guid.NewGuid();
			// PatientOrder
			Guid patientOrderTestOneGuid = Guid.NewGuid();
			// OrderedTest
			Guid orderedTestTestOneGuid = Guid.NewGuid();
			Guid orderedTestTestTwoGuid = Guid.NewGuid();
			// PatientSpecimen
			Guid patientSpecimenTestOneGuid = Guid.NewGuid();
			// No results 
			Guid patientSpecimenTestTwoGuid = Guid.NewGuid();
			// Specimen tests
			Guid specimenTestAboTestOneGuid = Guid.NewGuid();
			Guid specimenTestRhTestOneGuid = Guid.NewGuid();
			Guid specimenTestAboTestTwoGuid = Guid.NewGuid();
			Guid specimenTestRhTestTwoGuid = Guid.NewGuid();
			//													 
			// First, we need two new Specimen Uids
            string specimenUid1 = DataCreator.GetNewSpecimenUid();
            string specimenUid2 = DataCreator.GetNewSpecimenUid();
            DateTime currentTime = DateTime.Now;

			//
			StringBuilder query = new StringBuilder();
			//
			query.Append("BEGIN TRANSACTION T1 ");
			//
			query.Append("INSERT INTO PATIENTTREATMENT (");
			query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(currentTime)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTORDER (");
			query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
			query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(currentTime)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
            //
            // Specimen #1
            query.Append("INSERT INTO PATIENTSPECIMEN (");
            query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
            query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
            query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
            query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
            query.Append("VALUES (");
            query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
            query.Append("'").Append(patientGuid.ToString()).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(currentTime)).Append("',");
            query.Append("'A',");
            query.Append("'0',");
            query.Append("'").Append(specimenUid1).Append("',");
            query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(currentTime)).Append("',");
            query.Append("'UnitTestUser',");
            query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
            query.Append("\n");
            //
            // Specimen #2
            query.Append("INSERT INTO PATIENTSPECIMEN (");
            query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
            query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
            query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
            query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
            query.Append("VALUES (");
            query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
            query.Append("'").Append(patientGuid.ToString()).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(currentTime)).Append("',");
            query.Append("'A',");
            query.Append("'0',");
            query.Append("'").Append(specimenUid2).Append("',");
            query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(currentTime)).Append("',");
            query.Append("'UnitTestUser',");
            query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
            query.Append("\n");
            //
			query.Append("INSERT INTO ORDEREDTEST (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
            query.Append(TABLES.OrderedTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
            query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'1',");
			query.Append("'A',");
			query.Append("'P',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(currentTime)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO ORDEREDTEST (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientSpecimenGuid).Append(",");
            query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
            query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'1',");
			query.Append("'A',");
			query.Append("'P',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(currentTime)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");


			// 18: A
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.AboRhChangeIndicator).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAboTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'18',");
			query.Append("'A',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(currentTime)).Append("',");
			if (currentJustified) query.Append(1).Append(",");
			else query.Append(0).Append(",");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(currentTime)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			if(createRhTests)
			{
				// 21: P
				query.Append("INSERT INTO SPECIMENTEST (");
				query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
				query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
				query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
				query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
				query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
				query.Append(TABLES.SpecimenTest.TestDate).Append(",");
				query.Append(TABLES.SpecimenTest.AboRhChangeIndicator).Append(",");
				query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
				query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
				query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
				query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
				query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
				query.Append("VALUES (");
				query.Append("'").Append(specimenTestRhTestOneGuid.ToString()).Append("',");
				query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
				query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
				query.Append("'21',");
				query.Append("'P',");
				query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(currentTime)).Append("',");
				query.Append(0).Append(",");
				query.Append("'A',");
				query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
				query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(currentTime)).Append("',");
				query.Append("'UnitTestUser',");
				query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
				query.Append("\n");
			}

			//18: A
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.AboRhChangeJustification).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAboTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'18',");
			query.Append("'A',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(currentTime.AddDays(-1))).Append("',");
			query.Append(0).Append(",");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(currentTime.AddDays(-1))).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			if (createRhTests)
			{
				// 21: P or N
				query.Append("INSERT INTO SPECIMENTEST (");
				query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
				query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
				query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
				query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
				query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
				query.Append(TABLES.SpecimenTest.TestDate).Append(",");
				query.Append(TABLES.SpecimenTest.AboRhChangeJustification).Append(",");
				query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
				query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
				query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
				query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
				query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
				query.Append("VALUES (");
				query.Append("'").Append(specimenTestRhTestTwoGuid.ToString()).Append("',");
				query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
				query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
				query.Append("'21',");
				if (matching) query.Append("'P',");
				else query.Append("'N',");
                query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(currentTime.AddDays(-1))).Append("',");
				query.Append(0).Append(",");
				query.Append("'A',");
				query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
				query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(currentTime.AddDays(-1))).Append("',");
				query.Append("'UnitTestUser',");
				query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
				query.Append("\n");
			}
			//
			query.Append("COMMIT TRANSACTION T1 ");
			//
			UnitTestUtil.RunSQL(query.ToString());
		    query.Clear();

			#endregion
			
			BOL.Specimen specimen = new BOL.Specimen(patientSpecimenTestOneGuid);
			if (matching || currentJustified)
			{
				// The results match
				passed = (Common.TypingTestResults.Match == specimen.RepeatABORhTestsMatch);
			}
			else
			{
				if (!createRhTests)
				{
					passed = (Common.TypingTestResults.NoResults == specimen.RepeatABORhTestsMatch);
				}
				else
				{
					// The results don't match
					passed = (Common.TypingTestResults.Discrepancy == specimen.RepeatABORhTestsMatch);

					if (passed)
					{
						// There are no results
						specimen = new BOL.Specimen(patientSpecimenTestTwoGuid);
						
						Common.ABORhSource aboRhSource = specimen.Patient.ABORhSource;
						passed = (Common.ABORhSource.SpecimenTestDiscrepancy == aboRhSource);
					}
				}
			}

            if (!skipDelete)
            {
                //
                // Delete test data

                #region Clear Test Data

                query.Append("BEGIN TRANSACTION T2 ");
                query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestAboTestOneGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestRhTestOneGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestAboTestTwoGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestRhTestTwoGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM ORDEREDTEST WHERE OrderedTestGuid ='").Append(orderedTestTestOneGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM ORDEREDTEST WHERE OrderedTestGuid ='").Append(orderedTestTestTwoGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM PATIENTSPECIMEN WHERE PatientSpecimenGuid ='").Append(patientSpecimenTestOneGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM PATIENTSPECIMEN WHERE PatientSpecimenGuid ='").Append(patientSpecimenTestTwoGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM PATIENTORDER WHERE PatientOrderGuid ='").Append(patientOrderTestOneGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM PATIENTTREATMENT WHERE PatientTreatmentGuid ='").Append(patientTreatmentTestOneGuid).Append("' ").Append("\n");
                query.Append("COMMIT TRANSACTION T2 ");
                UnitTestUtil.RunSQL(query.ToString());

                #endregion

                // All done!

            }
		    return passed;
		}




        private bool TestABORhResultsSpecimen(Guid patientGuid, bool complete)
        {
            bool passed = false;
            // Create two specimen tests to be used to justify abo/rh change
            #region Create Test Data

            // PatientTreatment
            Guid patientTreatmentTestOneGuid = Guid.NewGuid();
            // PatientOrder
            Guid patientOrderTestOneGuid = Guid.NewGuid();
            // OrderedTest
            Guid orderedTestTestOneGuid = Guid.NewGuid();
            // PatientSpecimen
            Guid patientSpecimenTestOneGuid = Guid.NewGuid();
            // Specimen tests
            Guid specimenTestAboTestOneGuid = Guid.NewGuid();
            Guid specimenTestRhTestOneGuid = Guid.NewGuid();
            //													 
            // First, we need two new Specimen Uids
            string specimenUid1 = DataCreator.GetNewSpecimenUid();
            string specimenUid2 = DataCreator.GetNewSpecimenUid();
           
            //
            StringBuilder query = new StringBuilder();
            //
            query.Append("BEGIN TRANSACTION T1 ");
            //
            query.Append("INSERT INTO PATIENTTREATMENT (");
            query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
            query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
            query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
            query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
            query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
            query.Append("VALUES(");
            query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
            query.Append("'").Append(patientGuid.ToString()).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
            query.Append("'UnitTestUser',");
            query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
            query.Append("\n");
            //
            query.Append("INSERT INTO PATIENTORDER (");
            query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
            query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
            query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
            query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
            query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
            query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
            query.Append("VALUES(");
            query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
            query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
            query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
            query.Append("'UnitTestUser',");
            query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
            query.Append("\n");
            //
            // Specimen #1
            query.Append("INSERT INTO PATIENTSPECIMEN (");
            query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
            query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
            query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
            query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
            query.Append("VALUES (");
            query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
            query.Append("'").Append(patientGuid.ToString()).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
            query.Append("'A',");
            query.Append("'0',");
            query.Append("'").Append(specimenUid1).Append("',");
            query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
            query.Append("'UnitTestUser',");
            query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
            query.Append("\n");
            //

            query.Append("INSERT INTO ORDEREDTEST (");
            query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
            query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
            query.Append(TABLES.OrderedTest.PatientSpecimenGuid).Append(",");
            query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
            query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
            query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
            query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
            query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
            query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
            query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
            query.Append("VALUES (");
            query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
            query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
            query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
            query.Append("'1',");
            query.Append("'A',");
            query.Append("'P',");
            query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
            query.Append("'UnitTestUser',");
            query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
            query.Append("\n");
            //

           
            // 18: A
            query.Append("INSERT INTO SPECIMENTEST (");
            query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
            query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
            query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
            query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
            query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
            query.Append(TABLES.SpecimenTest.TestDate).Append(",");
            query.Append(TABLES.SpecimenTest.AboRhChangeIndicator).Append(",");
            query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
            query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
            query.Append("VALUES (");
            query.Append("'").Append(specimenTestAboTestOneGuid.ToString()).Append("',");
            query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
            query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
            query.Append("'18',");
            query.Append("'A',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
            query.Append(0).Append(",");
            query.Append("'A',");
            query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
            query.Append("'UnitTestUser',");
            query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
            query.Append("\n");

            if (complete)
            {
                // 21: P
                query.Append("INSERT INTO SPECIMENTEST (");
                query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
                query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
                query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
                query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
                query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
                query.Append(TABLES.SpecimenTest.TestDate).Append(",");
                query.Append(TABLES.SpecimenTest.AboRhChangeIndicator).Append(",");
                query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
                query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
                query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
                query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
                query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
                query.Append("VALUES (");
                query.Append("'").Append(specimenTestRhTestOneGuid.ToString()).Append("',");
                query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
                query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
                query.Append("'21',");
                query.Append("'P',");
                query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
                query.Append(0).Append(",");
                query.Append("'A',");
                query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
                query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
                query.Append("'UnitTestUser',");
                query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
                query.Append("\n");
            }

           
            //
            query.Append("COMMIT TRANSACTION T1 ");
            //
            UnitTestUtil.RunSQL(query.ToString());
            query.Remove(0, query.Length);

            #endregion

            BOL.Specimen specimen = new BOL.Specimen(patientSpecimenTestOneGuid);
            if (complete)
            {
                passed = specimen.ABORhTested;
            }
            else
            {
                passed = !specimen.ABORhTested;   
            }

            //
            // Delete test data
            #region Clear Test Data

            query.Append("BEGIN TRANSACTION T2 ");
            query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestAboTestOneGuid).Append("' ").Append("\n");
            query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestRhTestOneGuid).Append("' ").Append("\n");
            query.Append("DELETE FROM ORDEREDTEST WHERE OrderedTestGuid ='").Append(orderedTestTestOneGuid).Append("' ").Append("\n");
            query.Append("DELETE FROM PATIENTSPECIMEN WHERE PatientSpecimenGuid ='").Append(patientSpecimenTestOneGuid).Append("' ").Append("\n");
            query.Append("DELETE FROM PATIENTORDER WHERE PatientOrderGuid ='").Append(patientOrderTestOneGuid).Append("' ").Append("\n");
            query.Append("DELETE FROM PATIENTTREATMENT WHERE PatientTreatmentGuid ='").Append(patientTreatmentTestOneGuid).Append("' ").Append("\n");
            query.Append("COMMIT TRANSACTION T2 ");
            UnitTestUtil.RunSQL(query.ToString());

            #endregion
            // All done!
            return passed;
        }





		/// <summary>
		/// This unit test helper method creates test data for two different methods to help
		/// support
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <param name="matching"></param>
		/// <param name="justified"></param>
		private bool CreateTestDataDifferentSpecimens(Guid patientGuid, bool matching, bool justified, bool pastAboJustified, bool pastRhJustified, bool skipDelete)
		{
			bool passed = false;
			// Create two specimen tests to be used to justify abo/rh change
			#region Create Test Data

			Guid patientTreatmentTestOneGuid = Guid.NewGuid();
			Guid patientTreatmentTestTwoGuid = Guid.NewGuid();
			Guid patientOrderTestOneGuid = Guid.NewGuid();
			Guid patientOrderTestTwoGuid = Guid.NewGuid();
			Guid orderedTestTestOneGuid = Guid.NewGuid();
			Guid orderedTestTestTwoGuid = Guid.NewGuid();
			Guid patientSpecimenTestOneGuid = Guid.NewGuid();
			Guid patientSpecimenTestTwoGuid = Guid.NewGuid();
			// No results specimen
			Guid patientSpecimenTestThreeGuid = Guid.NewGuid();
			Guid specimenTestAboTestOneGuid = Guid.NewGuid();
			Guid specimenTestRhTestOneGuid = Guid.NewGuid();
			Guid specimenTestAboTestTwoGuid = Guid.NewGuid();
			Guid specimenTestRhTestTwoGuid = Guid.NewGuid();
			//													 
			// First, we need two new Specimen Uids
            string specimenUid1 = DataCreator.GetNewSpecimenUid();
            string specimenUid2 = DataCreator.GetNewSpecimenUid();
            string specimenUid3 = DataCreator.GetNewSpecimenUid();
			//
			StringBuilder query = new StringBuilder();
			//
			query.Append("BEGIN TRANSACTION T1 ");
			//
			query.Append("INSERT INTO PATIENTTREATMENT (");
			query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTTREATMENT (");
			query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientTreatmentTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTORDER (");
			query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
			query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTORDER (");
			query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
			query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientOrderTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientTreatmentTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
            //
            // Specimen #1
            query.Append("INSERT INTO PATIENTSPECIMEN (");
            query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
            query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
            query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
            query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
            query.Append("VALUES (");
            query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
            query.Append("'").Append(patientGuid.ToString()).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
            query.Append("'A',");
            query.Append("'0',");
            query.Append("'").Append(specimenUid1).Append("',");
            query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
            query.Append("'UnitTestUser',");
            query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
            query.Append("\n");
            //
            // Specimen #2
            query.Append("INSERT INTO PATIENTSPECIMEN (");
            query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
            query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
            query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
            query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
            query.Append("VALUES (");
            query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
            query.Append("'").Append(patientGuid.ToString()).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
            query.Append("'A',");
            query.Append("'0',");
            query.Append("'").Append(specimenUid2).Append("',");
            query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
            query.Append("'UnitTestUser',");
            query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
            query.Append("\n");
            //
            // Specimen #3
            query.Append("INSERT INTO PATIENTSPECIMEN (");
            query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
            query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
            query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
            query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
            query.Append("VALUES (");
            query.Append("'").Append(patientSpecimenTestThreeGuid.ToString()).Append("',");
            query.Append("'").Append(patientGuid.ToString()).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
            query.Append("'A',");
            query.Append("'0',");
            query.Append("'").Append(specimenUid3).Append("',");
            query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
            query.Append("'UnitTestUser',");
            query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
            query.Append("\n");

			query.Append("INSERT INTO ORDEREDTEST (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
            query.Append(TABLES.OrderedTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
            query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'1',");
			query.Append("'A',");
			query.Append("'P',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO ORDEREDTEST (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
            query.Append(TABLES.OrderedTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderTestTwoGuid.ToString()).Append("',");
            query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'1',");
			query.Append("'A',");
			query.Append("'P',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// 18: A
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.AboRhChangeIndicator).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAboTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'18',");
			query.Append("'A',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			if (justified) query.Append(1).Append(",");
			else query.Append(0).Append(",");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			// 21: P
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.AboRhChangeIndicator).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestRhTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'21',");
			query.Append("'P',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			if (justified) query.Append(1).Append(",");
			else query.Append(0).Append(",");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			//18: A
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.AboRhChangeIndicator).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAboTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'18',");
			//if (matching) query.Append("'A',");
			//else query.Append("'B',");
			query.Append("'A',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-1))).Append("',");
			if (pastAboJustified) query.Append(1).Append(",");
			else query.Append(0).Append(",");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-1))).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			// 21: P or N
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.AboRhChangeIndicator).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestRhTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'21',");
			if (matching) query.Append("'P',");
			else query.Append("'N',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-1))).Append("',");
			if (pastRhJustified) query.Append(1).Append(",");
			else query.Append(0).Append(",");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(-1))).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//

			query.Append("COMMIT TRANSACTION T1 ");
			//
			UnitTestUtil.RunSQL(query.ToString());
		    query.Clear();

			#endregion
			
			BOL.Specimen specimen = new BOL.Specimen(patientSpecimenTestOneGuid);
			if (matching || justified || pastAboJustified || pastRhJustified)
			{
				// The results match
				//Assertion.AssertEquals("ABORhTestsMatchHistoricResults-Match", Common.TypingTestResults.Match , specimen.ABORhTestsMatchHistoricResults());
				Common.ABORhSource aboRhSource = specimen.Patient.ABORhSource;
				passed = (aboRhSource == Common.ABORhSource.SpecimenTest ||
					(aboRhSource == Common.ABORhSource.SpecimenTestDiscrepancy && pastAboJustified) ||
					(aboRhSource == Common.ABORhSource.SpecimenTestDiscrepancy && pastRhJustified));
				
				
			}
			else
			{
				// The results don't match
				//Assertion.AssertEquals("ABORhTestsMatchHistoricResults-Discrepancy", Common.TypingTestResults.Discrepancy , specimen.ABORhTestsMatchHistoricResults());
				Common.ABORhSource aboRhSource = specimen.Patient.ABORhSource;
				passed = (aboRhSource == Common.ABORhSource.SpecimenTestDiscrepancy);
				//Assertion.AssertEquals("ABORhTestsMatchHistoricResults-Discrepancy", Common.ABORhSource.SpecimenTestDiscrepancy, aboRhSource);

				// There are no results
				specimen = new BOL.Specimen(patientSpecimenTestThreeGuid);
				//Assertion.AssertEquals("ABORhTestsMatchHistoricResults-NoResults", Common.TypingTestResults.NoResults , specimen.ABORhTestsMatchHistoricResults());
				//aboRhSource = specimen.Patient.ABORhSource;
				//Assertion.AssertEquals("ABORhTestsMatchHistoricResults-NoResults", Common.ABORhSource.None, aboRhSource);

			}

            if (!skipDelete)
            {
                //
                // Delete test data

                #region Clear Test Data

                query.Append("BEGIN TRANSACTION T2 ");
                query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestAboTestOneGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestRhTestOneGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestAboTestTwoGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestRhTestTwoGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM ORDEREDTEST WHERE OrderedTestGuid ='").Append(orderedTestTestOneGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM ORDEREDTEST WHERE OrderedTestGuid ='").Append(orderedTestTestTwoGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM PATIENTSPECIMEN WHERE PatientSpecimenGuid ='").Append(patientSpecimenTestOneGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM PATIENTSPECIMEN WHERE PatientSpecimenGuid ='").Append(patientSpecimenTestTwoGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM PATIENTSPECIMEN WHERE PatientSpecimenGuid ='").Append(patientSpecimenTestThreeGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM PATIENTORDER WHERE PatientOrderGuid ='").Append(patientOrderTestOneGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM PATIENTORDER WHERE PatientOrderGuid ='").Append(patientOrderTestTwoGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM PATIENTTREATMENT WHERE PatientTreatmentGuid ='").Append(patientTreatmentTestOneGuid).Append("' ").Append("\n");
                query.Append("DELETE FROM PATIENTTREATMENT WHERE PatientTreatmentGuid ='").Append(patientTreatmentTestTwoGuid).Append("' ").Append("\n");
                query.Append("COMMIT TRANSACTION T2 ");
                UnitTestUtil.RunSQL(query.ToString());

                #endregion

            }
		    // All done!
			return passed;
		}

		/// <summary>
		/// This unit test helper method creates test data for two different methods to help
		/// support - Private Method names cannot start with Test (to avoid confusing NUnit)
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <param name="matching"></param>
		/// <param name="justified"></param>
		private void CheckInconclusives(Guid patientGuid, bool matching, bool justified, bool pastAboJustified, bool pastRhJustified)
		{
			// Create two specimen tests to be used to justify abo/rh change
			#region Create Test Data

			Guid patientTreatmentTestOneGuid = Guid.NewGuid();
			Guid patientTreatmentTestTwoGuid = Guid.NewGuid();
			Guid patientOrderTestOneGuid = Guid.NewGuid();
			Guid patientOrderTestTwoGuid = Guid.NewGuid();
			Guid orderedTestTestOneGuid = Guid.NewGuid();
			Guid orderedTestTestTwoGuid = Guid.NewGuid();
			Guid patientSpecimenTestOneGuid = Guid.NewGuid();
			Guid patientSpecimenTestTwoGuid = Guid.NewGuid();
			// No results specimen
			Guid patientSpecimenTestThreeGuid = Guid.NewGuid();
			Guid specimenTestAboTestOneGuid = Guid.NewGuid();
			Guid specimenTestRhTestOneGuid = Guid.NewGuid();
			Guid specimenTestAboTestTwoGuid = Guid.NewGuid();
			Guid specimenTestRhTestTwoGuid = Guid.NewGuid();

			Guid specimenTestITestTwoGuid = Guid.NewGuid();
			//													 
			// First, we need two new Specimen Uids
            string specimenUid1 = DataCreator.GetNewSpecimenUid();
            string specimenUid2 = DataCreator.GetNewSpecimenUid();
            string specimenUid3 = DataCreator.GetNewSpecimenUid();
			//
			StringBuilder query = new StringBuilder();
			//
			query.Append("BEGIN TRANSACTION T1 ");
			//
			query.Append("INSERT INTO PATIENTTREATMENT (");
			query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTTREATMENT (");
			query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientTreatmentTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTORDER (");
			query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
			query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTORDER (");
			query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
			query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientOrderTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientTreatmentTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
            //
            // Specimen #1
            query.Append("INSERT INTO PATIENTSPECIMEN (");
            query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
            query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
            query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
            query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
            query.Append("VALUES (");
            query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
            query.Append("'").Append(patientGuid.ToString()).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
            query.Append("'A',");
            query.Append("'0',");
            query.Append("'").Append(specimenUid1).Append("',");
            query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
            query.Append("'UnitTestUser',");
            query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
            query.Append("\n");
            //
            // Specimen #2
            query.Append("INSERT INTO PATIENTSPECIMEN (");
            query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
            query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
            query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
            query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
            query.Append("VALUES (");
            query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
            query.Append("'").Append(patientGuid.ToString()).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
            query.Append("'A',");
            query.Append("'0',");
            query.Append("'").Append(specimenUid2).Append("',");
            query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
            query.Append("'UnitTestUser',");
            query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
            query.Append("\n");
            //
            // Specimen #3
            query.Append("INSERT INTO PATIENTSPECIMEN (");
            query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
            query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
            query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
            query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
            query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
            query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
            query.Append("VALUES (");
            query.Append("'").Append(patientSpecimenTestThreeGuid.ToString()).Append("',");
            query.Append("'").Append(patientGuid.ToString()).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
            query.Append("'A',");
            query.Append("'0',");
            query.Append("'").Append(specimenUid3).Append("',");
            query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
            query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
            query.Append("'UnitTestUser',");
            query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
            query.Append("\n");
            //
			query.Append("INSERT INTO ORDEREDTEST (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
            query.Append(TABLES.OrderedTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
            query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'1',");
			query.Append("'A',");
			query.Append("'P',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO ORDEREDTEST (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
            query.Append(TABLES.OrderedTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderTestTwoGuid.ToString()).Append("',");
            query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'1',");
			query.Append("'A',");
			query.Append("'P',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// I
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.AboRhChangeIndicator).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAboTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'18',");
			query.Append("'I',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			if (justified) query.Append(1).Append(",");
			else query.Append(0).Append(",");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			// 21: P
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.AboRhChangeIndicator).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestRhTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'21',");
			query.Append("'P',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			if (justified) query.Append(1).Append(",");
			else query.Append(0).Append(",");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			//18: A
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.AboRhChangeIndicator).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAboTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'18',");
			query.Append("'A',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			if (pastAboJustified) query.Append(1).Append(",");
			else query.Append(0).Append(",");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			// 21: P or N
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.AboRhChangeIndicator).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestRhTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'21',");
			if (matching) query.Append("'P',");
			else query.Append("'N',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			if (pastRhJustified) query.Append(1).Append(",");
			else query.Append(0).Append(",");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//

			

			query.Append("COMMIT TRANSACTION T1 ");
			//
			UnitTestUtil.RunSQL(query.ToString());
		    query.Clear();

			#endregion
			
			BOL.Specimen specimen = new BOL.Specimen(patientSpecimenTestOneGuid);
			if (matching || justified || pastAboJustified || pastRhJustified)
			{
				// The results match
				//Assertion.AssertEquals("ABORhTestsMatchHistoricResults-Match", Common.TypingTestResults.Match , specimen.ABORhTestsMatchHistoricResults());
				Common.ABORhSource aboRhSource = specimen.Patient.ABORhSource;
			}
			else
			{
				// The results don't match
				//Assertion.AssertEquals("ABORhTestsMatchHistoricResults-Discrepancy", Common.TypingTestResults.Discrepancy , specimen.ABORhTestsMatchHistoricResults());
				Common.ABORhSource aboRhSource = specimen.Patient.ABORhSource;
				Assert.AreEqual(Common.ABORhSource.ConversionDiscrepancy, aboRhSource, "ABORhTestsMatchHistoricResults-Discrepancy");

				// There are no results
				specimen = new BOL.Specimen(patientSpecimenTestThreeGuid);
				//Assertion.AssertEquals("ABORhTestsMatchHistoricResults-NoResults", Common.TypingTestResults.NoResults , specimen.ABORhTestsMatchHistoricResults());
				aboRhSource = specimen.Patient.ABORhSource;
				Assert.AreEqual(Common.ABORhSource.None, aboRhSource, "ABORhTestsMatchHistoricResults-NoResults");
			}
			//
			// Delete test data
			#region Clear Test Data

			query.Append("BEGIN TRANSACTION T2 ");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestAboTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestRhTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestAboTestTwoGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestRhTestTwoGuid).Append("' ").Append("\n");

			query.Append("DELETE FROM SPECIMENTEST WHERE SpecimenTestGuid ='").Append(specimenTestITestTwoGuid).Append("' ").Append("\n");
            query.Append("DELETE FROM ORDEREDTEST WHERE OrderedTestGuid ='").Append(orderedTestTestOneGuid).Append("' ").Append("\n");
            query.Append("DELETE FROM ORDEREDTEST WHERE OrderedTestGuid ='").Append(orderedTestTestTwoGuid).Append("' ").Append("\n");

			query.Append("DELETE FROM PATIENTSPECIMEN WHERE PatientSpecimenGuid ='").Append(patientSpecimenTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTSPECIMEN WHERE PatientSpecimenGuid ='").Append(patientSpecimenTestTwoGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTSPECIMEN WHERE PatientSpecimenGuid ='").Append(patientSpecimenTestThreeGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTORDER WHERE PatientOrderGuid ='").Append(patientOrderTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTORDER WHERE PatientOrderGuid ='").Append(patientOrderTestTwoGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTTREATMENT WHERE PatientTreatmentGuid ='").Append(patientTreatmentTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTTREATMENT WHERE PatientTreatmentGuid ='").Append(patientTreatmentTestTwoGuid).Append("' ").Append("\n");
			query.Append("COMMIT TRANSACTION T2 ");
			UnitTestUtil.RunSQL(query.ToString());

			#endregion
			// All done!
		}
	}
}
#endif

